//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: List.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 10-13-97
// Last modified: 10-13-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This class template encapsulates a doubly-linked list of any type.
//
// TYPE       Type of object stored in the list.
// ARG_TYPE   Type used to reference objects stored in the list.
//            Can be a reference.

#ifndef __LIST_H__
#define __LIST_H__

#include <Decls.h>

// POSITION type used for iteration
typedef void* POSITION;

template< class TYPE, class ARG_TYPE >
class CList
{
// Operations
public:
  CList( );
  CList( const CList< TYPE, ARG_TYPE >& rOther );
  ~CList( );

  CList< TYPE, ARG_TYPE >& operator=(
    const CList< TYPE, ARG_TYPE >& rOther );

  int GetCount( ) const;
  BOOL IsEmpty( ) const;
  
  void AddHead( ARG_TYPE item );
  TYPE GetHead( ) const;
  void RemoveHead( );

  void AddTail( ARG_TYPE item );
  TYPE GetTail( ) const;
  void RemoveTail( );

  void RemoveAt( POSITION pos );

  POSITION GetHeadPosition( ) const;
  POSITION GetTailPosition( ) const;
  TYPE GetNext( POSITION& pos ) const;
  TYPE GetPrev( POSITION& pos ) const;

  void RemoveAll( );
  
#ifndef NDEBUG
  void AssertValid( ) const;
  BOOL BelongsToList( POSITION pos ) const;
#endif

// Helpers
private:
  void Copy( const CList< TYPE, ARG_TYPE >& rOther );

// Attributes
private:
  class CCell
  {
  public:
    CCell( ARG_TYPE value, CCell* pPrev, CCell* pNext ) :
      m_value( value ), m_pPrev( pPrev ), m_pNext( pNext ) { };
    TYPE m_value;
    CCell* m_pPrev;
    CCell* m_pNext;
  };

  CCell* m_pFrontCell;
  CCell* m_pBackCell;
  int m_nSize;
};

/////////////////////////////////////////////////////////////////////////////
// CList< TYPE, ARG_TYPE > inline functions

// Return the number of elements the list contains
template< class TYPE, class ARG_TYPE >
inline int CList< TYPE, ARG_TYPE >::GetCount( ) const 
{
  ASSERT_VALID( this );

  return m_nSize;
}

// Return TRUE if the list is empty, otherwise return FALSE
template< class TYPE, class ARG_TYPE >
inline BOOL CList< TYPE, ARG_TYPE >::IsEmpty( ) const
{
  ASSERT_VALID( this );
  
  return m_nSize == 0;
}

// Return the first element of the list
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetHead( ) const
{
  ASSERT( m_nSize );
  ASSERT_VALID( this );

  return m_pFrontCell->m_value;
}

// Return the last element of the list
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetTail( ) const
{
  ASSERT( m_nSize );
  ASSERT_VALID( this );

  return m_pBackCell->m_value;
}

// Return the POSITION of the front element
template< class TYPE, class ARG_TYPE >
inline POSITION CList< TYPE, ARG_TYPE >::GetHeadPosition( ) const
{
  ASSERT_VALID( this );

  return m_pFrontCell;
}

// Return the POSITION of the last element
template< class TYPE, class ARG_TYPE >
inline POSITION CList< TYPE, ARG_TYPE >::GetTailPosition( ) const
{
  ASSERT_VALID( this );

  return m_pBackCell;
}

// Pop all elements (private)
template< class TYPE, class ARG_TYPE >
inline void CList< TYPE, ARG_TYPE >::RemoveAll( )
{
  while ( !IsEmpty( ) )
    RemoveHead( );

  ASSERT( IsEmpty( ) );
}

/////////////////////////////////////////////////////////////////////////////
// CList< TYPE, ARG_TYPE > out-of-line functions

// Default constructor
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >::CList( )
{
  m_nSize = 0;
  m_pFrontCell = NULL;
  m_pBackCell = NULL;

  ASSERT_VALID( this );
}

// Copy constructor
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >::CList( 
  const CList< TYPE, ARG_TYPE >& rOther )
{
  ASSERT_VALID( &rOther );

  Copy( rOther );
  
  ASSERT_VALID( this );
}

// Destructor
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >::~CList( )
{
  ASSERT_VALID( this );
  
  RemoveAll( );
}

// Assignment operator
template< class TYPE, class ARG_TYPE >
CList< TYPE, ARG_TYPE >& CList< TYPE, ARG_TYPE >::operator=(
  const CList< TYPE, ARG_TYPE >& rOther )
{
  ASSERT_VALID( this );
  ASSERT_VALID( &rOther );
  ASSERT( &rOther != this );
  
  RemoveAll( );
  Copy( rOther );

  ASSERT_VALID( this );

  return *this;
}

// Add 'item' to the front of the list
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AddHead( ARG_TYPE item )
{
  ASSERT_VALID( this );

  CCell* pNewCell = new CCell( item, NULL, m_pFrontCell );
  if ( m_pFrontCell )
    m_pFrontCell->m_pPrev = pNewCell;
  else
    m_pBackCell = pNewCell;
  m_pFrontCell = pNewCell;
    
  m_nSize++;

  ASSERT_VALID( this );
}

// Remove front element
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::RemoveHead( )
{
  ASSERT_VALID( this );
  ASSERT( m_nSize > 0 );

  CCell* m_pOldFrontCell = m_pFrontCell;
  m_pFrontCell = m_pFrontCell->m_pNext;
  delete m_pOldFrontCell;
  
  if ( m_pFrontCell )
    m_pFrontCell->m_pPrev = NULL;
  else
    m_pBackCell = NULL;

  m_nSize--;

  ASSERT_VALID( this );
}

// Add 'item' to the back of the list
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AddTail( ARG_TYPE item )
{
  ASSERT_VALID( this );

  CCell* pNewCell = new CCell( item, m_pBackCell, NULL );
  if ( m_pBackCell )
    m_pBackCell->m_pNext = pNewCell;
  else
    m_pFrontCell = pNewCell;
  m_pBackCell = pNewCell;
  
  m_nSize++;

  ASSERT_VALID( this );
}

// Remove back element
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::RemoveTail( )
{
  ASSERT_VALID( this );
  ASSERT( m_nSize > 0 );
  
  CCell* pOldBackCell = m_pBackCell;
  m_pBackCell = m_pBackCell->m_pPrev;
  delete pOldBackCell;
  
  if ( m_pBackCell )
    m_pBackCell->m_pNext = NULL;
  else
    m_pFrontCell = NULL;

  m_nSize--;

  ASSERT_VALID( this );
}

// Return element at pos and move to next position
// pos will be NULL if there are no more elements
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetNext( POSITION& pos ) const
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  CCell* pCell = ( CCell* ) pos;
  pos = pCell->m_pNext;

  return pCell->m_value;
}

// Return element at pos and move to previous position
// pos will be NULL if there are no more elements
template< class TYPE, class ARG_TYPE >
inline TYPE CList< TYPE, ARG_TYPE >::GetPrev( POSITION& pos ) const
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  CCell* pCell = ( CCell* ) pos;
  pos = pCell->m_pPrev;

  return pCell->m_value;
}

// Remove element at pos
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::RemoveAt( POSITION pos )
{
  ASSERT_VALID( this );
  ASSERT( BelongsToList( pos ) );

  CCell* pCell = ( ( CCell* ) pos );

  if ( pCell->m_pPrev )
    pCell->m_pPrev->m_pNext = pCell->m_pNext;
  else
    m_pFrontCell = pCell->m_pNext;

  if ( pCell->m_pNext )
    pCell->m_pNext->m_pPrev = pCell->m_pPrev;
  else
    m_pBackCell = pCell->m_pPrev;

  delete pos;
  
  m_nSize--;

  ASSERT_VALID( this );
}


// Private function: copy rOther into this
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::Copy( const CList< TYPE, ARG_TYPE >& rOther )
{
  // Set up empty list
  m_nSize = 0;
  m_pFrontCell = NULL;
  m_pBackCell = NULL;

  // Copy all elements of rOther
  CCell* pCell = rOther.m_pFrontCell;
  while ( pCell )
  {
    AddTail( pCell->m_value );
    pCell = pCell->m_pNext;
  }
}

// Check class invariant (debug version only)
#ifndef NDEBUG
template< class TYPE, class ARG_TYPE >
void CList< TYPE, ARG_TYPE >::AssertValid( ) const
{
  if ( m_nSize == 0 )
    ASSERT( m_pFrontCell == NULL && m_pBackCell == NULL );
  else if ( m_nSize == 1 )
    ASSERT( m_pFrontCell == m_pBackCell && m_pBackCell->m_pNext == NULL 
            && m_pFrontCell->m_pPrev == NULL );
  else
  {
    int i;
    CCell* pCell;

    pCell = m_pFrontCell;
    for ( i = 0; i < m_nSize - 1; i++ )
      pCell = pCell->m_pNext;
    ASSERT( pCell == m_pBackCell );
    ASSERT( m_pBackCell->m_pNext == NULL );
    
    pCell = m_pBackCell;
    for ( i = 0; i < m_nSize - 1; i++ )
      pCell = pCell->m_pPrev;
    ASSERT( pCell == m_pFrontCell );
    ASSERT( m_pFrontCell->m_pPrev == NULL );
  }
}

// Return TRUE if pos is a POSITION of this list else return FALSE
// (debug version only)
template< class TYPE, class ARG_TYPE >
BOOL CList< TYPE, ARG_TYPE >::BelongsToList( POSITION pos ) const
{
  CCell* pCell = m_pFrontCell;
  while ( pCell )
  {
    if ( pCell == pos )
      return TRUE;
    pCell = pCell->m_pNext;
  }

  return FALSE;
}

#endif

#endif // __LIST_H__
